<?php
/**
 * Copyright © 2015 Magento. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace UseePay\Afterpay\Model;

use Magento\Payment\Model\Method\AbstractMethod;
use Magento\Quote\Api\Data\CartInterface;
use Magento\Sales\Model\Order;

class PaymentMethod extends AbstractMethod
{
    const CODE = 'useepayafterpay';
    const POST = "[POST to UseePay]";

    protected $_code = self::CODE;

    protected $_isInitializeNeeded = true;

    protected $_formBlockType = 'UseePay\Afterpay\Block\Form';
    protected $_infoBlockType = 'UseePay\Afterpay\Block\Info';

    protected $_isGateway = false;
    protected $_canAuthorize = false;
    protected $_canCapture = false;
    protected $_canCapturePartial = false;
    protected $_canRefund = false;
    protected $_canRefundInvoicePartial = false;
    protected $_canVoid = false;
    protected $_canUseInternal = false;
    protected $_canUseCheckout = true;
    protected $_canUseForMultishipping = false;
    protected $_canSaveCc = false;

    protected $urlBuilder;
    protected $_moduleList;
    protected $checkoutSession;
    protected $_orderFactory;


    public function __construct(
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory,
        \Magento\Payment\Helper\Data $paymentData,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Payment\Model\Method\Logger $logger,
        \Magento\Framework\Module\ModuleListInterface $moduleList,
        \Magento\Framework\Stdlib\DateTime\TimezoneInterface $localeDate,
        \Magento\Sales\Model\OrderFactory $orderFactory,
        \Magento\Framework\Url $urlBuilder,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
        \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
        array $data = []
    )
    {
        $this->urlBuilder = $urlBuilder;
        $this->_moduleList = $moduleList;
        $this->checkoutSession = $checkoutSession;
        $this->_orderFactory = $orderFactory;
        parent::__construct($context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $paymentData,
            $scopeConfig,
            $logger,
            $resource,
            $resourceCollection,
            $data);
    }

    /**
     *  Redirect URL
     *
     * @return   string Redirect URL
     */
    public function getOrderPlaceRedirectUrl()
    {
        return $this->urlBuilder->getUrl('useepayafterpay/payment/redirect', ['_secure' => true]);
    }

    /**
     *  failure URL
     *
     * @return   string failure URL
     */
    public function getPageFailureUrl()
    {
        return $this->urlBuilder->getUrl('checkout/onepage/failure', ['_secure' => true]);
    }

    public function updateOrderFail($orderId, $result)
    {
        $status_success = $this->getConfigData('success_order_status');    // 配置的成功状态
        $status_failure = $this->getConfigData('failure_order_status');    // 配置的失败状态
        $order = $this->_orderFactory->create()->loadByIncrementId($orderId);
        $status_history = $order->getState();
        if (empty($status_history) || ($status_history != $status_failure && $status_history != $status_success)) {
            $order->getPayment()
                ->setPreparedMessage('')
                ->setTransactionId($result['reference'])
                ->setIsTransactionClosed(0)
                ->update(false);
            $order->setState($status_failure);
            $order->setStatus($status_failure);
            $order->addStatusToHistory($status_failure, __('sorry, payment is failed. Result Message is: ' . $result["errorMsg"]));
            $order->save();
        }
    }

    /**
     *  Gateway URL
     *
     * @return   string Gateway URL
     */
    public function getGatewayUrl()
    {
        return $this->getConfigData('gateway_url');
    }

    public function canUseForCurrency($currencyCode)
    {
        return true;
    }

    public function initialize($paymentAction, $stateObject)
    {
        $payment = $this->getInfoInstance();
        //$order = $payment->getOrder();

        $state = $this->getConfigData('new_order_status');

        //$state = Mage_Sales_Model_Order::STATE_PENDING_PAYMENT;
        $stateObject->setState($state);
        $stateObject->setStatus('pending_payment');
        $stateObject->setIsNotified(false);
    }

    /**
     * 日志
     * @param $data
     */
    public function saveLog($data)
    {
        file_put_contents(dirname(dirname(__FILE__)) . "/log/" . "useepay.log", date("Y-m-d H:i:s", time()) . " " . $data . PHP_EOL, FILE_APPEND);
    }

    /**
     * 获取IP
     * @return mixed
     */
    public function getIP()
    {
        if (array_key_exists("HTTP_X_FORWARDED_FOR", $_SERVER))
            $ip = $_SERVER ["HTTP_X_FORWARDED_FOR"];
        else if (array_key_exists("HTTP_CLIENT_IP", $_SERVER))
            $ip = $_SERVER ["HTTP_CLIENT_IP"];
        else if (array_key_exists("HTTP_X_REAL_IP", $_SERVER))
            $ip = $_SERVER ["HTTP_X_REAL_IP"];
        else if (array_key_exists("REMOTE_ADDR", $_SERVER))
            $ip = $_SERVER ["REMOTE_ADDR"];
        else if (@getenv("HTTP_X_FORWARDED_FOR"))
            $ip = getenv("HTTP_X_FORWARDED_FOR");
        else if (@getenv("HTTP_CLIENT_IP"))
            $ip = getenv("HTTP_CLIENT_IP");
        else if (@getenv("HTTP_X_REAL_IP"))
            $ip = getenv("HTTP_X_REAL_IP");
        else if (@getenv("REMOTE_ADDR"))
            $ip = getenv("REMOTE_ADDR");
        else
            $ip = "0.0.0.0";
        $ip = explode(', ',$ip);
        $ip = $ip['0'];
        return $ip;
    }

    /**
     * 生成签名串
     * @param $arr
     * @return string
     */
    public function createMd5Sign($arr, $secretKey)
    {
        if (empty($arr) || empty($secretKey)) {
            return false;
        }
        $str = '';
        ksort($arr);
        foreach ($arr as $key => $value) {
            $v = trim($value);
            if (strlen($v) > 0 && $key != 'sign') {
                $str .= '&' . $key . '=' . $v;
            }
        }
        $str = substr($str, 1);
        $str .= "&pkey=" . trim($secretKey);
        return strtolower(md5($str));
    }

    /**
     * 获取终端类型
     * @return string
     */
    public function getTerminalType()
    {
        $terminalType = 'WEB';
        $useragent = $_SERVER['HTTP_USER_AGENT'];
        if (preg_match('/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows (ce|phone)|xda|xiino/i', $useragent) || preg_match('/1207|6310|6590|3gso|4thp|50[1-6]i|770s|802s|a wa|abac|ac(er|oo|s\-)|ai(ko|rn)|al(av|ca|co)|amoi|an(ex|ny|yw)|aptu|ar(ch|go)|as(te|us)|attw|au(di|\-m|r |s )|avan|be(ck|ll|nq)|bi(lb|rd)|bl(ac|az)|br(e|v)w|bumb|bw\-(n|u)|c55\/|capi|ccwa|cdm\-|cell|chtm|cldc|cmd\-|co(mp|nd)|craw|da(it|ll|ng)|dbte|dc\-s|devi|dica|dmob|do(c|p)o|ds(12|\-d)|el(49|ai)|em(l2|ul)|er(ic|k0)|esl8|ez([4-7]0|os|wa|ze)|fetc|fly(\-|_)|g1 u|g560|gene|gf\-5|g\-mo|go(\.w|od)|gr(ad|un)|haie|hcit|hd\-(m|p|t)|hei\-|hi(pt|ta)|hp( i|ip)|hs\-c|ht(c(\-| |_|a|g|p|s|t)|tp)|hu(aw|tc)|i\-(20|go|ma)|i230|iac( |\-|\/)|ibro|idea|ig01|ikom|im1k|inno|ipaq|iris|ja(t|v)a|jbro|jemu|jigs|kddi|keji|kgt( |\/)|klon|kpt |kwc\-|kyo(c|k)|le(no|xi)|lg( g|\/(k|l|u)|50|54|\-[a-w])|libw|lynx|m1\-w|m3ga|m50\/|ma(te|ui|xo)|mc(01|21|ca)|m\-cr|me(rc|ri)|mi(o8|oa|ts)|mmef|mo(01|02|bi|de|do|t(\-| |o|v)|zz)|mt(50|p1|v )|mwbp|mywa|n10[0-2]|n20[2-3]|n30(0|2)|n50(0|2|5)|n7(0(0|1)|10)|ne((c|m)\-|on|tf|wf|wg|wt)|nok(6|i)|nzph|o2im|op(ti|wv)|oran|owg1|p800|pan(a|d|t)|pdxg|pg(13|\-([1-8]|c))|phil|pire|pl(ay|uc)|pn\-2|po(ck|rt|se)|prox|psio|pt\-g|qa\-a|qc(07|12|21|32|60|\-[2-7]|i\-)|qtek|r380|r600|raks|rim9|ro(ve|zo)|s55\/|sa(ge|ma|mm|ms|ny|va)|sc(01|h\-|oo|p\-)|sdk\/|se(c(\-|0|1)|47|mc|nd|ri)|sgh\-|shar|sie(\-|m)|sk\-0|sl(45|id)|sm(al|ar|b3|it|t5)|so(ft|ny)|sp(01|h\-|v\-|v )|sy(01|mb)|t2(18|50)|t6(00|10|18)|ta(gt|lk)|tcl\-|tdg\-|tel(i|m)|tim\-|t\-mo|to(pl|sh)|ts(70|m\-|m3|m5)|tx\-9|up(\.b|g1|si)|utst|v400|v750|veri|vi(rg|te)|vk(40|5[0-3]|\-v)|vm40|voda|vulc|vx(52|53|60|61|70|80|81|83|85|98)|w3c(\-| )|webc|whit|wi(g |nc|nw)|wmlb|wonu|x700|yas\-|your|zeto|zte\-/i', substr($useragent, 0, 4))) {
            $terminalType = 'WAP';
        }
        return $terminalType;
    }

    /**
     * 获取终端类型
     * @return string
     */
    public function getIOSOrAndroid()
    {
        $terminalType = 'ANDROID';
        $useragent = $_SERVER['HTTP_USER_AGENT'];
        if (preg_match('/(iPod|iPad|iPhone)/', $useragent)) {
            $terminalType = 'IOS';
        }
        return $terminalType;
    }

    /**
     * price转成最小单位级别
     * @param $currency
     * @param $price
     * @return int
     */
    public function getIntPrice($currency, $price)
    {
        $currency_map_json = '{"EUR":2,"AMD":2,"BOV":2,"XAF":0,"COP":2,"COU":2,"CUP":2,"ANG":2,"DOP":2,"HKD":2,"IRR":2,"NPR":2,"PYG":0,"QAR":2,"RON":2,"SBD":2,"SOS":2,"LKR":2,"TWD":2,"VUV":0,"XTS":0,"XAU":0,"AOA":2,"ARS":2,"BRL":2,"BGN":2,"KHR":2,"KMF":0,"CDF":2,"GNF":0,"HNL":2,"ISK":0,"ILS":2,"JPY":0,"KWD":3,"LBP":2,"LYD":3,"MWK":2,"MXV":2,"PKR":2,"RWF":0,"STD":2,"SLL":2,"CHW":2,"UGX":0,"USN":2,"UZS":2,"XBB":0,"BBD":2,"XOF":0,"BMD":2,"CAD":2,"NZD":2,"CZK":2,"FJD":2,"GBP":2,"ZAR":2,"LRD":2,"PHP":2,"WST":2,"SCR":2,"SDG":2,"TMT":2,"UYI":0,"XBA":0,"BAM":2,"BWP":2,"CLP":0,"CRC":2,"GYD":2,"JOD":3,"MKD":2,"MUR":2,"MDL":2,"MNT":2,"NGN":2,"SRD":2,"SEK":2,"TZS":2,"TOP":2,"UAH":2,"UYU":2,"ZWL":2,"USD":2,"BSD":2,"BYN":2,"BZD":2,"CLF":4,"GHS":2,"GTQ":2,"IQD":3,"KRW":0,"MOP":2,"MXN":2,"MAD":2,"NIO":2,"XSU":0,"SZL":2,"THB":2,"VND":0,"ZMW":2,"XPD":0,"AFN":2,"AUD":2,"BOB":2,"CNY":2,"HRK":2,"DJF":0,"EGP":2,"ETB":2,"GEL":2,"HTG":2,"XDR":0,"JMD":2,"KZT":2,"KPW":2,"MGA":2,"MVR":2,"XUA":0,"MMK":2,"NAD":2,"OMR":3,"PGK":2,"RUB":2,"TTD":2,"XXX":0,"ALL":2,"DZD":2,"XCD":2,"AZN":2,"BDT":2,"INR":2,"BTN":2,"NOK":2,"BND":2,"BIF":0,"CUC":2,"SVC":2,"FKP":2,"XPF":0,"GMD":2,"HUF":2,"KES":2,"KGS":2,"LAK":2,"MYR":2,"MRO":2,"PAB":2,"PEN":2,"RSD":2,"SGD":2,"CHE":2,"TND":3,"TRY":2,"VEF":2,"YER":2,"XBD":0,"XPT":0,"XAG":0,"AWG":2,"BHD":3,"CVE":2,"KYD":2,"DKK":2,"ERN":2,"GIP":2,"IDR":2,"LSL":2,"CHF":2,"MZN":2,"PLN":2,"SHP":2,"SAR":2,"SSP":2,"SYP":2,"TJS":2,"AED":2,"XBC":0}';

        $currency_map = json_decode($currency_map_json, true);
        $times = 2;
        if (array_key_exists($currency, $currency_map)) {
            $times = $currency_map[$currency];
        }
        $newPrice = round(floatval($price), $times);
        while ($times--) {
            $newPrice *= 10;
        }
        return intval(strval($newPrice));
    }

    /**
     * 转换特殊字符串
     * @param $str
     * @return mixed
     */
    public function format($str)
    {
        //转义"双引号,<小于号,>大于号,'单引号
        return str_replace(
            ["<", ">", "'", "\"", "&", "\r", "\n", "\r\n"],
            ["", "", "", "", "", " ", " ", " "],
            trim($str));
    }

    /**
     * 获取商品详情
     */
    function getProductItems($AllItems)
    {
        $productDetails = [];

        foreach ($AllItems as $item) {
            $tmp = [];
            $tmp['name'] = $item->getName();
            $tmp['body'] = $item->getName();
            $tmp['quantity'] = intval($item->getQtyOrdered());
            $tmp['price'] = $item->getPrice();
            $productDetails[] = $tmp;
        }
        return $productDetails;
    }

    /**
     * 构建请求参数
     * @return array
     */
    public function getCheckoutParameter()
    {
        $merchant_id = $this->getConfigData('partner_id');
        $appId      = $this->getConfigData('app_id');
        $order_id = $this->checkoutSession->getLastRealOrderId();
        $order = $this->_orderFactory->create()->loadByIncrementId($order_id);
        $currency = $order->getOrderCurrencyCode();
        $order_amount = $this->getIntPrice($currency, $order->getGrandTotal());
        $billing = $order->getBillingAddress();
        $country = $billing->getCountryId();
        $pkey = $this->getConfigData('security_code');
        $customer_ip = $this->getIP();
        $notify_url = $this->urlBuilder->getUrl('useepayafterpay/payment/notice', ['_secure' => true, '_nosid' => true]);
        $redirect_url = $this->urlBuilder->getUrl('useepayafterpay/payment/back', ['_secure' => true, '_nosid' => true]);
        $billingAddress  = [];
        $shippingAddress = [];
        $orderBillingAddress  = $order->getBillingAddress();
        $orderShippingAddress = $order->getShippingAddress();

        if (isset($orderBillingAddress)) {
            if (sizeof($orderBillingAddress->getStreet()) > 1) {
                $billingAddress['houseNo'] = $orderBillingAddress->getStreet()[1];
            } else {
                $billingAddress['houseNo'] = '';
            }
            if (sizeof($orderBillingAddress->getStreet()) > 0) {
                $billingAddress['street'] = $orderBillingAddress->getStreet()[0];
            } else {
                $billingAddress['street'] = '';
            }
            $billingAddress['email']      = $orderBillingAddress->getEmail();
            $billingAddress['phoneNo']    = $orderBillingAddress->getTelephone();
            $billingAddress['firstName']  = $orderBillingAddress->getFirstname();
            $billingAddress['lastName']   = $orderBillingAddress->getLastname();
            $billingAddress['postalCode'] = $orderBillingAddress->getPostcode();
            $billingAddress['city']       = $orderBillingAddress->getCity();
            $billingAddress['state']      = $orderBillingAddress->getRegion();
            $billingAddress['country']    = $orderBillingAddress->getCountryId();
        }
        if (isset($orderShippingAddress)) {
            if (sizeof($orderShippingAddress->getStreet()) > 1) {
                $shippingAddress['houseNo'] = $orderShippingAddress->getStreet()[1];
            } else {
                $shippingAddress['houseNo'] = '';
            }
            if (sizeof($orderShippingAddress->getStreet()) > 0) {
                $shippingAddress['street'] = $orderShippingAddress->getStreet()[0];
            } else {
                $shippingAddress['street'] = '';
            }
            $shippingAddress['email'] = $orderShippingAddress->getEmail();
            $shippingAddress['phoneNo'] = $orderShippingAddress->getTelephone();
            $shippingAddress['firstName'] = $orderShippingAddress->getFirstname();
            $shippingAddress['lastName'] = $orderShippingAddress->getLastname();
            $shippingAddress['postalCode'] = $orderShippingAddress->getPostcode();
            $shippingAddress['city'] = $orderShippingAddress->getCity();
            $shippingAddress['state'] = $orderShippingAddress->getRegion();
            $shippingAddress['country'] = $orderShippingAddress->getCountryId();

            if (!isset($billingAddress)) {
                if (sizeof($orderShippingAddress->getStreet()) > 1) {
                    $billingAddress['houseNo'] = $orderShippingAddress->getStreet()[0];
                } else {
                    $billingAddress['houseNo'] = '';
                }
                if (sizeof($orderShippingAddress->getStreet()) > 0) {
                    $billingAddress['street'] = $orderShippingAddress->getStreet()[1];
                } else {
                    $billingAddress['street'] = '';
                }
                $billingAddress['email'] = $orderShippingAddress->getEmail();
                $billingAddress['phoneNo'] = $orderShippingAddress->getTelephone();
                $billingAddress['firstName'] = $orderShippingAddress->getFirstname();
                $billingAddress['lastName'] = $orderShippingAddress->getLastname();
                $billingAddress['postalCode'] = $orderShippingAddress->getPostcode();
                $billingAddress['city'] = $orderShippingAddress->getCity();
                $billingAddress['state'] = $orderShippingAddress->getRegion();
                $billingAddress['country'] = $orderShippingAddress->getCountryId();
            }
        }
        $payerInfo = [
            'paymentMethod'       => 'afterpay',
            'billingAddress'      => $billingAddress
        ];

        $payerInfo = json_encode($payerInfo);
        $userInfo = json_encode(
            [
                'email'   => empty($billingAddress['email']) ? $shippingAddress['email'] : $billingAddress['email'],
                'phoneNo' => empty($billingAddress['phoneNo']) ? $shippingAddress['phoneNo'] : $billingAddress['phoneNo'],
                'ip'      => $customer_ip
            ]
        );
        $productList = $this->getProductItems($order->getAllItems());
        $subject     = '';
        foreach ($productList as $product) {
            $subject .= $product['name'];
        }
        if (empty($subject)) {
            $subject = 'Goods';
        }
        $orderInfo = json_encode([
            'subject'         => $subject,
            'goodsInfo'       => $productList,
            'shippingAddress' => $shippingAddress
        ]);

        $params         = [
            'version'                   => '1.0',
            'merchantNo'                => $merchant_id,
            'transactionType'           => 'pay',
            'transactionId'             => $order_id,
            'transactionExpirationTime' => '144',
            'appId'                     => $appId,
            'currency'                  => $currency,
            'amount'                    => $order_amount,
            'notifyUrl'                 => $notify_url,
            'redirectUrl'               => $redirect_url,
            'reserved'                  => json_encode([
                'pluginName'    => 'Magento2-AfterPay',
                'pluginVersion' => '1.0.1'
            ]),
            'signType'                  => 'MD5',
            'payerInfo'                 => $payerInfo,
            'userInfo'                  => $userInfo,
            'orderInfo'                 => $orderInfo,
            'autoRedirect'              => 'false'
        ];
        $params['sign'] = $this->createMD5Sign($params, $pkey);

        $this->saveLog('request url is: ' . $this->getConfigData('gateway_url'));
        $this->saveLog('param is:');
        $this->saveLog(var_export($params, true));
        $this->saveLog("--------------------------------------");

        return $params;
    }
}
