<?php

namespace UseePay\Payments\Controller\Payment;

use Magento\Checkout\Model\Session;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Lock\LockManagerInterface;
use Magento\Framework\Url;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Email\Sender\OrderSender;
use Magento\Sales\Model\OrderFactory;
use UseePay\Payments\service\InvoiceService;
use UseePay\Payments\Controller\AbsUseePayAction;
use UseePay\Payments\Model\Adminhtml\Source\Config;
use UseePay\Payments\Helper\SignatureHelper;
use UseePay\Payments\Helper\OrderStatusHelper;
use Magento\Framework\Lock\Backend\Cache as CacheLock;
use UseePay\Payments\service\LockService;


class SyncResult extends AbsUseePayAction implements HttpPostActionInterface
{
    /**
     * @var Config
     */
    private $config;
    /**
     * @var JsonFactory
     */
    private $jsonFactory;
    /**
     * @var OrderFactory
     */
    private $orderFactory;
    /**
     * @var Url
     */
    private $urlBuilder;
    /**
     * @var OrderSender
     */
    private $orderSender;
    /**
     * @var Session
     */
    private $checkoutSession;
    const TRANSACTION_TYPE_ORDER = 'order';

    /**
     * @var \Magento\Quote\Api\CartRepositoryInterface
     */
    protected $quoteRepository;
    protected $lockService;

    private $invoiceService;

    public function __construct(Context $context, Config $config,
                                JsonFactory $jsonFactory, OrderFactory $orderFactory,
                                Url $urlBuilder, OrderSender $orderSender,Session $checkoutSession,
                                \Magento\Quote\Api\CartRepositoryInterface $quoteRepository = null,
                                LockService $lockService,
                                InvoiceService $invoiceService

    )
    {
        parent::__construct($context);
        $this->config       = $config;
        $this->jsonFactory  = $jsonFactory;
        $this->orderFactory = $orderFactory;
        $this->urlBuilder   = $urlBuilder;
        $this->orderSender  = $orderSender;
        $this->checkoutSession    = $checkoutSession;
        $this->quoteRepository = $quoteRepository ?: ObjectManager::getInstance()
            ->get(\Magento\Quote\Api\CartRepositoryInterface::class);
        $this->lockService    = $lockService;
        $this->invoiceService    = $invoiceService;
    }

    public function execute() {
        $result   = json_decode($this->getRequest()->getContent(), true);
        $response = $this->jsonFactory->create();
        $data     = [];
        $this->saveLog('response-SyncNotify:');
        $this->saveLog(var_export($result, true));
        $this->saveLog('------------------------------');


        if (SignatureHelper::verifyMd5Signature($result, $this->config->getMd5SecretKey())) {
            $order = $this->orderFactory->create()->loadByIncrementId($result['transactionId']);
            $status_history = $order->getState();

            if (OrderStatusHelper::isSucceed($result)) {
                $this->paySuccess($order, $status_history, $result);
                $url             = 'checkout/onepage/success';
                $data['success'] = true;
                $data['message'] = 'payment is successful.';
            } else if (OrderStatusHelper::isFailed($result)) {
                $this->payFailed($status_history, $order, $result);
                $url             = '';
                $data['success'] = false;
                $data['message'] = 'payment failed! Result Message is: ' . $result["errorMsg"];
            } else {
                $this->payPending($order, $result, $status_history);
                $url             = 'checkout/onepage/success';
                $data['success'] = true;
                $data['message'] = 'payment is processing,  Result Message is: ' . $result["errorMsg"];
            }
            $url                 = $this->urlBuilder->getUrl($url);
            $data['redirectUrl'] = $url;
        } else {
            $data['success'] = false;
            $data['message'] = 'Invalid signature.';
        }
        $response->setData($data);
        return $response;
    }





    /**
     * @param $order
     * @param $status_history
     * @param $result
     * @return void
     */
    public function paySuccess($order, $status_history, $result): void
    {
        $quote = $this->quoteRepository->get($order->getQuoteId())->setIsActive(false);
        $this->quoteRepository->save($quote);

        if (OrderStatusHelper::canChangeStatus($status_history)) {
            $this->saveLog('syncResult paySuccess $status_history: ' . $status_history);
            $lock = $this->lockService->lock($result['transactionId']);

            try {
                if ($lock) {
                    $order = $this->orderFactory->create()->loadByIncrementId($result['transactionId']);
                    $status_history = $order->getState();
                    if (OrderStatusHelper::canChangeStatus($status_history)) {
                        $order->getPayment()
                            ->setPreparedMessage('')
                            ->setTransactionId($result['reference'])
                            ->setIsTransactionClosed(0)
                            ->update(false);
                        $order->setState(Order::STATE_PROCESSING);
                        $order->setStatus(Order::STATE_PROCESSING);
                        $order->addStatusToHistory(Order::STATE_PROCESSING, __('payment is successful.'));
                        $order->setTotalPaid($order->getGrandTotal());
                        $order->setBaseTotalPaid($order->getGrandTotal());
                        $order->save();
                        $this->invoiceService->createOrderInvoice($order, $result, true);
                        $this->orderSender->send($order);
                    }
                }
            } finally {
                if ($lock) {
                    $this->lockService->unlock($result['transactionId']);
                }
            }
        }
    }

    /**
     * @param $status_history
     * @param $order
     * @param $errorMsg
     * @return void
     */
    public function payFailed($status_history, $order, $result): void
    {
        // After the order is successful, the status cannot be changed to 'failed' again
        $this->saveLog('syncResult payFailed $status_history : ' . $status_history);

        $transactionId = $result['transactionId'];
        $errorMsg = $result['errorMsg'];

        if (OrderStatusHelper::canChangeStatus($status_history)) {
            $lock = $this->lockService->lock($transactionId);
            try {
                if ($lock) {
                    $order = $this->orderFactory->create()->loadByIncrementId($result['transactionId']);
                    $status_history = $order->getState();
                    if (OrderStatusHelper::canChangeFailStatus($status_history)) {
                        $order->setState(Order::STATE_CANCELED);
                        $order->setStatus(Order::STATE_CANCELED);
                        $order->addStatusToHistory(Order::STATE_CANCELED, __('sorry, payment is failed! Result Message is: ' . $errorMsg));
                        $order->save();
                        $errorMsg = $result["errorMsg"];
                        $this->messageManager->addError(__('payment failed! Result Message is: ' . $errorMsg));
                    }
                }
            } finally {
                $this->lockService->unlock($transactionId);
            }
        }
    }

    /**
     * @param $order
     * @param $result
     * @param $status_history
     * @return void
     */
    public function payPending($order, $result, $status_history): void
    {

        // After the order is successful, the status cannot be changed to 'failed' again
        $this->saveLog('syncResult payPending $status_history : ' . $status_history);
        $transactionId = $result['transactionId'];

        if (OrderStatusHelper::canChangeStatus($status_history)) {
            $lock = $this->lockService->lock($transactionId);
            try {
                if ($lock) {
                    $order = $this->orderFactory->create()->loadByIncrementId($transactionId);
                    $status_history = $order->getState();
                    if (OrderStatusHelper::canChangeStatus($status_history)) {
                        $quote = $this->quoteRepository->get($order->getQuoteId())->setIsActive(false);
                        $this->quoteRepository->save($quote);
                        $order->getPayment()
                            ->setPreparedMessage('')
                            ->setTransactionId($result['reference'])
                            ->setIsTransactionClosed(0)
                            ->update(false);
                        $this->messageManager->addNoticeMessage(__('payment is processing! Result Message is: ' . $result["errorMsg"]));

                        $order->setState(Order::STATE_PAYMENT_REVIEW);
                        $order->setStatus(Order::STATE_PAYMENT_REVIEW);
                        $order->addStatusToHistory(Order::STATE_PAYMENT_REVIEW, __('payment is processing,  Result Message is: ' . $result["errorMsg"]));
                        $order->save();
                    }
                }
            } finally {
                $this->lockService->unlock($transactionId);
            }
        }
    }
}
