<?php

namespace UseePay\Payments\Controller\Payment;

use Magento\Checkout\Model\Cart;
use Magento\Checkout\Model\Session;
use Magento\Customer\Api\Data\GroupInterface;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\HttpRequestInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Event\ManagerInterface as EventManager;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Url;
use Magento\Payment\Helper\Data;
use Magento\Quote\Api\CartManagementInterface;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Model\CustomerManagement;
use Magento\Quote\Model\Quote;
use Magento\Quote\Model\Quote as QuoteEntity;
use Magento\Quote\Model\Quote\Address\ToOrder as ToOrderConverter;
use Magento\Quote\Model\Quote\Address\ToOrderAddress as ToOrderAddressConverter;
use Magento\Quote\Model\Quote\Item\ToOrderItem as ToOrderItemConverter;
use Magento\Quote\Model\Quote\Payment\ToOrderPayment as ToOrderPaymentConverter;
use Magento\Quote\Model\QuoteFactory;
use Magento\Quote\Model\QuoteIdMaskFactory;
//use Magento\Quote\Model\SubmitQuoteValidator;
use Magento\Sales\Api\OrderManagementInterface as OrderManagement;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\OrderFactory;
use UseePay\Payments\Controller\AbsUseePayAction;
use UseePay\Payments\Model\Adminhtml\Source\Config;
use UseePay\Payments\Helper\CurrencyHelper;
use UseePay\Payments\Helper\SignatureHelper;
use UseePay\Payments\Helper\PlaceOrderHelper;
class Token extends AbsUseePayAction implements HttpRequestInterface
{
    /**
     * Checkout types: Checkout as Guest
     */
    const METHOD_GUEST = 'guest';
    /**
     * @var Config
     */
    private $config;
    /**
     * @var Session
     */
    private $checkoutSession;
    /**
     * @var OrderFactory
     */
    private $orderFactory;
    /**
     * @var Url
     */
    private $urlBuilder;
    /**
     * @var Cart
     */
    private $cart;
    /**
     * @var CartManagementInterface
     */
    private $quoteManagement;
    /**
     * @var \Magento\Customer\Model\Session
     */
    private $customerSession;
    /**
     * @var QuoteFactory
     */
    private $quoteFactory;
    /**
     * @var CartRepositoryInterface
     */
    private $quoteRepository;
    private $jsonResultFactory;
    /**
     * @var Data
     */
    private $_paymentHelper;
    /**
     * @var \Magento\Payment\Model\Config
     */
    private $_paymentConfig;
    /**
     * @var Magento\Quote\Model\QuoteIdMaskFactory
     */
    private $quoteIdMaskFactory;

    /**
     * @var \Magento\Framework\HTTP\PhpEnvironment\RemoteAddress
     */
    private $remoteAddress;
    /**
     * @var \Magento\Framework\App\RequestInterface
     */
    private $request;

    /**
     * @var \Magento\Customer\Api\AddressRepositoryInterface
     */
    private $addressRepository;

    /**
     * @var EventManager
     */
    protected $eventManager;

    /**
     * @var \Magento\Sales\Api\Data\OrderInterfaceFactory
     */
    protected $orderInterfaceFactory;

//    /**
//     * @var SubmitQuoteValidator
//     */
//    private $submitQuoteValidator;

    /**
     * @var CustomerManagement
     */
    protected $customerManagement;

    /**
     * @var \Magento\Framework\Api\DataObjectHelper
     */
    protected $dataObjectHelper;

    /**
     * @var ToOrderConverter
     */
    protected $quoteAddressToOrder;
    /**
     * @var ToOrderAddressConverter
     */
    protected $quoteAddressToOrderAddress;

    /**
     * @var ToOrderPaymentConverter
     */
    protected $quotePaymentToOrderPayment;
    /**
     * @var ToOrderItemConverter
     */
    protected $quoteItemToOrderItem;

    /**
     * @var OrderManagement
     */
    protected $orderManagement;

    /**
     * @var \Magento\Customer\Api\CustomerRepositoryInterface
     */
    protected $customerRepository;

    /**
     * @var PlaceOrderHelper
     */
    private $placeOrderHelper;

    public function __construct(
        Context $context,
        Config $config,
        Url $urlBuilder,
        \Magento\Customer\Model\Session $customerSession,
        Session $checkoutSession,
        OrderFactory $orderFactory,
        QuoteFactory $quoteFactory,
        Cart $cart,
        CartManagementInterface $quoteManagement,
        CartRepositoryInterface $quoteRepository,
        JsonFactory $jsonResultFactory,
        Data $paymentHelper,
        \Magento\Payment\Model\Config $paymentConfig,
        QuoteIdMaskFactory $quoteIdMaskFactory,

        \Magento\Framework\HTTP\PhpEnvironment\RemoteAddress $remoteAddress = null,
        \Magento\Framework\App\RequestInterface $request = null,
        \Magento\Customer\Api\AddressRepositoryInterface $addressRepository = null,
//        EventManager $eventManager
        \Magento\Sales\Api\Data\OrderInterfaceFactory $orderInterfaceFactory = null,
//        SubmitQuoteValidator $submitQuoteValidator = null,
        CustomerManagement $customerManagement = null,
        \Magento\Framework\Api\DataObjectHelper $dataObjectHelper = null,
        ToOrderConverter $quoteAddressToOrder = null,
        ToOrderAddressConverter $quoteAddressToOrderAddress = null,
        ToOrderPaymentConverter $quotePaymentToOrderPayment = null,
        ToOrderItemConverter $quoteItemToOrderItem = null,
        OrderManagement $orderManagement = null,
        \Magento\Customer\Api\CustomerRepositoryInterface $customerRepository = null,
        PlaceOrderHelper $placeOrderHelper
    )
    {
        parent::__construct($context);
        $this->config             = $config;
        $this->customerSession    = $customerSession;
        $this->checkoutSession    = $checkoutSession;
        $this->orderFactory       = $orderFactory;
        $this->quoteFactory       = $quoteFactory;
        $this->urlBuilder         = $urlBuilder;
        $this->cart               = $cart;
        $this->quoteManagement    = $quoteManagement;
        $this->quoteRepository    = $quoteRepository;
        $this->jsonResultFactory  = $jsonResultFactory;
        $this->_paymentHelper     = $paymentHelper;
        $this->_paymentConfig     = $paymentConfig;
        $this->quoteIdMaskFactory = $quoteIdMaskFactory;

        $this->remoteAddress = $remoteAddress ?: ObjectManager::getInstance()
            ->get(\Magento\Framework\HTTP\PhpEnvironment\RemoteAddress::class);
        $this->request = $request ?: ObjectManager::getInstance()
            ->get(\Magento\Framework\App\RequestInterface::class);
        $this->addressRepository = $addressRepository ?: ObjectManager::getInstance()
            ->get(\Magento\Customer\Api\AddressRepositoryInterface::class);
//        $this->eventManager = $eventManager;
        $this->orderInterfaceFactory = $orderInterfaceFactory ?: ObjectManager::getInstance()
            ->get(\Magento\Sales\Api\Data\OrderInterfaceFactory::class);
//        $this->submitQuoteValidator = $submitQuoteValidator ?: ObjectManager::getInstance()
//            ->get(\Magento\Quote\Model\SubmitQuoteValidator::class);
        $this->customerManagement = $customerManagement ?: ObjectManager::getInstance()
            ->get(CustomerManagement::class);
        $this->dataObjectHelper = $dataObjectHelper ?: ObjectManager::getInstance()
            ->get(\Magento\Framework\Api\DataObjectHelper::class);
        $this->quoteAddressToOrder = $quoteAddressToOrder ?: ObjectManager::getInstance()
            ->get(ToOrderConverter::class);
        $this->quoteAddressToOrderAddress = $quoteAddressToOrderAddress ?: ObjectManager::getInstance()
            ->get(ToOrderAddressConverter::class);
        $this->quotePaymentToOrderPayment = $quotePaymentToOrderPayment ?: ObjectManager::getInstance()
            ->get(ToOrderPaymentConverter::class);
        $this->quoteItemToOrderItem = $quoteItemToOrderItem ?: ObjectManager::getInstance()
            ->get(ToOrderItemConverter::class);
        $this->orderManagement = $orderManagement ?: ObjectManager::getInstance()
            ->get(OrderManagement::class);
        $this->customerRepository = $customerRepository ?: ObjectManager::getInstance()
            ->get(\Magento\Customer\Api\CustomerRepositoryInterface::class);
        $this->placeOrderHelper   = $placeOrderHelper;
    }

    /**
     * 日志
     * @param $data
     */
    public function saveLog($data)
    {
        file_put_contents(dirname(__FILE__, 3) . "/log/" . "UseePay.log", date("Y-m-d H:i:s", time()) . " " . $data . PHP_EOL, FILE_APPEND);
    }

    /**
     * Get all payment methods
     *
     * @return array
     */
    public function getAllPaymentMethods()
    {
        return $this->_paymentHelper->getPaymentMethods();
    }

    /**
     * Get key-value pair of all payment methods
     * key = method code & value = method name
     *
     * @return array
     */
    public function getAllPaymentMethodsList()
    {
        return $this->_paymentHelper->getPaymentMethodList();
    }

    /**
     * Get active/enabled payment methods
     *
     * @return array
     */
    public function getActivePaymentMethods()
    {
        return $this->_paymentConfig->getActiveMethods();
    }

    public function execute()
    {

        $quoteId    = '';
        $guestEmail = '';
        if (!empty($this->getRequest()->getContent())) {
            $requestParams = json_decode($this->getRequest()->getContent(), true);
            $quote = $this->checkoutSession->getQuote();
            $quoteId       = $quote->getId();
            $guestEmail    = $requestParams['email'];
            $this->saveLog('Request quoteId:' . $quoteId);
            $this->saveLog('Request email:' . $guestEmail);
        }
        try {
            $params = $this->buildParams($quoteId, $guestEmail);
        } catch (\Exception $e) {
            $this->saveLog('Exception:' . $e->getMessage());
            $response = $this->jsonResultFactory->create();
            $data['success'] = false;
            $data['message'] = $e->getMessage();
            $response->setData($data);
            return $response;
        }

        $this->saveLog('Request Params:\n' . var_export($params, true));
        $response = $this->jsonResultFactory->create();
        $data     = [];
        try {
            $result = json_decode($this->sendRequest($this->config->getEndpoint(), $params), true);
            $this->saveLog('Request result:\n' . var_export($result, true));

            if (!empty($result) && is_array($result) && !empty($result['token'])) {
                $data['success'] = true;
                $data['token']   = $result['token'];
            } else {
                $data['success'] = false;
                $data['message'] = $result['errorMsg'];
            }
        } catch (\Exception $e) {
            $data['success'] = false;
            $data['message'] = 'Network error, please try later!';
        }

        $response->setData($data);

        return $response;
    }


    private function getUseePayPaymentMethod()
    {
        foreach ($this->getAllPaymentMethods() as $payment) {

            try {
                if (substr($payment['title'], 0, strlen('useepay')) == 'UseePay') {
                    $this->saveLog('Payment:' . "\n" . var_export($payment, true));
                    return $payment;
                }
            } catch (\Exception $e) {
            }
        }
    }

    private function buildParams($quoteId, $guestEmail)
    {
        $quote = $this->cart->getQuote();

        // Create an Order ID for the customer's quote
//            $quote->reserveOrderId()->save(); // Warning: The may cause order ID skipping if the customer abandons the checkout

        if (empty($quote) || $quote->isEmpty() || empty($quote->getAllItems())) {
            try {
                $quote = $this->quoteRepository->get($quoteId);
            } catch (NoSuchEntityException $e) {
                $this->saveLog('Exception:' . "\n" . $e->getMessage());
                $quoteIdMask = $this->quoteIdMaskFactory->create()->load($quoteId, 'masked_id');
                $this->saveLog('Quote Id from Masked:' . $quoteIdMask->getQuoteId());
                $quote = $this->quoteRepository->get($quoteIdMask->getQuoteId());
            }
            $order = $this->orderFactory->create()->loadByIncrementId($quote->getReservedOrderId());
        } else {
            $incrementId = $this->checkoutSession->getLastRealOrder()->getIncrementId();
            $this->saveLog('incrementId:' . $incrementId);
            $status_history = null;
            $order = null;

            if ($incrementId) {
                $order = $this->orderFactory->create()->loadByIncrementId($incrementId);
                $status_history = $order->getState();
            }
            $this->saveLog('status_history:' . $status_history);
            $this->saveLog('quote grandTotal:' . $quote->getGrandTotal());
            if ($order) {
                $this->saveLog('order grandTotal:' . $order->getGrandTotal());
                $this->saveLog('payment:' . var_export($order->getPayment()->getMethod(), true));
            }
            $quote->setTotalsCollectedFlag(false);
            $quote->collectTotals();
            $quote->getPayment()->importData(['method' => 'useepay_payments']);
            // Set Checkout Method
            if (!$this->customerSession->isLoggedIn()) {
                // Use Guest Checkout
                $quote->setCheckoutMethod(self::METHOD_GUEST)
                    ->setCustomerId(null)
                    ->setCustomerEmail(empty($guestEmail) ? $quote->getBillingAddress()->getEmail() : $guestEmail)
                    ->setCustomerIsGuest(true)
                    ->setCustomerGroupId(\Magento\Customer\Api\Data\GroupInterface::NOT_LOGGED_IN_ID);
            } else {
                $quote
                    ->setCheckoutMethod(\Magento\Checkout\Model\Type\Onepage::METHOD_CUSTOMER);
            }
            $this->quoteRepository->save($quote);
            if ($order == null || ($status_history != Order::STATE_CANCELED || $quote->getGrandTotal() != $order->getGrandTotal())) {
                $orderId = $this->placeOrder($quoteId, $guestEmail);
                $order = $this->_objectManager->create(\Magento\Sales\Api\OrderRepositoryInterface::class)->get($orderId);
            } else {
                $order->setQuoteId($quote->getId());
                $order->save();
            }
        }

        if ($order == null) {
            return [];
        }

        $this->saveLog('Order Id:' . $quote->getReservedOrderId());

        $this->saveLog('quote Id:' . $quote->getId());

        $merchantNo = $this->config->getMerchantNo();
        $appId      = $this->config->getAppId();
//            $this->saveLog('CheckoutSession:\n' . var_export($this->checkoutSession->getLastRealOrder()->getData(), true));

        $currency = $order->getOrderCurrencyCode();
        $amount   = CurrencyHelper::convertPriceToUseePayPrice($currency, $order->getGrandTotal());
        $this->saveLog('Order Grand Total:\n' . var_export($order->getGrandTotal(), true));
        $this->saveLog('Order Base GrandTotal:\n' . var_export($order->getBaseGrandTotal(), true));
        $this->saveLog('Order Base G:\n' . var_export($order->getTotalQtyOrdered(), true));
        $country         = $order->getBillingAddress()->getCountryId();
        $terminalType    = $this->getTerminalType();
        $billingAddress  = [];
        $shippingAddress = [];

        $orderBillingAddress  = $order->getBillingAddress();
        $orderShippingAddress = $order->getShippingAddress();
        if (isset($orderBillingAddress)) {
            if (sizeof($orderBillingAddress->getStreet()) > 1) {
                $billingAddress['houseNo'] = $orderBillingAddress->getStreet()[1];
            } else {
                $billingAddress['houseNo'] = '';
            }
            if (sizeof($orderBillingAddress->getStreet()) > 0) {
                $billingAddress['street'] = $orderBillingAddress->getStreet()[0];
            } else {
                $billingAddress['street'] = '';
            }
            $billingAddress['email']      = $orderBillingAddress->getEmail();
            $billingAddress['phoneNo']    = $orderBillingAddress->getTelephone();
            $billingAddress['firstName']  = $orderBillingAddress->getFirstname();
            $billingAddress['lastName']   = $orderBillingAddress->getLastname();
            $billingAddress['postalCode'] = $orderBillingAddress->getPostcode();
            $billingAddress['city']       = $orderBillingAddress->getCity();
            $billingAddress['state']      = $orderBillingAddress->getRegion();
            $billingAddress['country']    = $orderBillingAddress->getCountryId();
        }
        if (isset($orderShippingAddress)) {
            if (sizeof($orderShippingAddress->getStreet()) > 1) {
                $shippingAddress['houseNo'] = $orderShippingAddress->getStreet()[1];
            } else {
                $shippingAddress['houseNo'] = '';
            }
            if (sizeof($orderShippingAddress->getStreet()) > 0) {
                $shippingAddress['street'] = $orderShippingAddress->getStreet()[0];
            } else {
                $shippingAddress['street'] = '';
            }
            $shippingAddress['email']      = $orderShippingAddress->getEmail();
            $shippingAddress['phoneNo']    = $orderShippingAddress->getTelephone();
            $shippingAddress['firstName']  = $orderShippingAddress->getFirstname();
            $shippingAddress['lastName']   = $orderShippingAddress->getLastname();
            $shippingAddress['postalCode'] = $orderShippingAddress->getPostcode();
            $shippingAddress['city']       = $orderShippingAddress->getCity();
            $shippingAddress['state']      = $orderShippingAddress->getRegion();
            $shippingAddress['country']    = $orderShippingAddress->getCountryId();

            if (!isset($billingAddress)) {
                if (sizeof($orderShippingAddress->getStreet()) > 1) {
                    $billingAddress['houseNo'] = $orderShippingAddress->getStreet()[0];
                } else {
                    $billingAddress['houseNo'] = '';
                }
                if (sizeof($orderShippingAddress->getStreet()) > 0) {
                    $billingAddress['street'] = $orderShippingAddress->getStreet()[1];
                } else {
                    $billingAddress['street'] = '';
                }
                $billingAddress['email']      = $orderShippingAddress->getEmail();
                $billingAddress['phoneNo']    = $orderShippingAddress->getTelephone();
                $billingAddress['firstName']  = $orderShippingAddress->getFirstname();
                $billingAddress['lastName']   = $orderShippingAddress->getLastname();
                $billingAddress['postalCode'] = $orderShippingAddress->getPostcode();
                $billingAddress['city']       = $orderShippingAddress->getCity();
                $billingAddress['state']      = $orderShippingAddress->getRegion();
                $billingAddress['country']    = $orderShippingAddress->getCountryId();
            }

            $payerInfo = json_encode([
                'paymentMethod'       => 'credit_card',
                'authorizationMethod' => 'cvv',
                'billingAddress'      => $billingAddress

            ]);

            $userInfo = json_encode(
                [
                    'email'   => empty($billingAddress['email']) ? $shippingAddress['email'] : $billingAddress['email'],
                    'phoneNo' => empty($billingAddress['phoneNo']) ? $shippingAddress['phoneNo'] : $billingAddress['phoneNo'],
                    'ip'      => $this->getCustomerIp()
                ]
            );

            $productList = $this->getProductItems($order->getAllItems());
            $subject     = '';
            foreach ($productList as $product) {
                $subject .= $product['name'];
            }

            $orderInfo = json_encode([
                'subject'         => $subject,
                'goodsInfo'       => $productList,
                'shippingAddress' => $shippingAddress
            ]);

            $params         = [
                'version'                   => '1.0',
                'merchantNo'                => $merchantNo,
                'transactionType'           => 'pay',
                'transactionId'             => $order->getIncrementId(),
                'transactionExpirationTime' => '30',
                'appId'                     => $appId,
                'currency'                  => $currency,
                'amount'                    => $amount,
                'country'                   => $country,
                'language'                  => 'en',
                'terminalType'              => $terminalType,
                'notifyUrl'                 => $this->urlBuilder->getUrl('useepay/payment/asyncResult', ['_secure' => true, '_nosid' => true]),
                'reserved'                  => json_encode([
                    'pluginName'    => 'Magento2',
                    'pluginVersion' => '1.0.2'
                ]),
                'charset'                   => 'utf-8',
                'signType'                  => 'MD5',
                'payerInfo'                 => $payerInfo,
                'userInfo'                  => $userInfo,
                'orderInfo'                 => $orderInfo,
                'autoRedirect'              => 'false'
            ];
            $params['sign'] = SignatureHelper::createMd5Signature($params, $this->config->getMd5SecretKey());
        }
        return $params;
    }


    /**
     * 获取IP
     * @return mixed
     */
    private function getCustomerIp()
    {
        $remoteAddr = $this->remoteAddress->getRemoteAddress();
        $ips = explode(",", $remoteAddr);
        return $ips[0];
    }

    private function getProductItems($allItems)
    {
        $productDetails = [];

        foreach ($allItems as $item) {
            $tmp              = [];
            $tmp['name']      = $item->getName();
            $tmp['quantity']  = intval($item->getQtyOrdered());
            $tmp['price']     = $item->getPrice();
            $productDetails[] = $tmp;
        }
        return $productDetails;
    }

    public function isPost()
    {
        return true;
    }

    public function isGet()
    {
        return true;
    }

    public function isPatch()
    {
        return false;
    }

    public function isDelete()
    {
        return false;
    }

    public function isPut()
    {
        return false;
    }

    public function isAjax()
    {
        return false;
    }

    public function placeOrder($cartId, $guestEmail)
    {
        $quote = $this->quoteRepository->getActive($cartId);
        $quote->collectTotals();

        if ($quote->getCheckoutMethod() === self::METHOD_GUEST) {
            $quote->setCustomerId(null);
            $quote->setCustomerEmail(empty($guestEmail) ? $quote->getBillingAddress()->getEmail() : $guestEmail);
            if ($quote->getCustomerFirstname() === null && $quote->getCustomerLastname() === null) {
                $quote->setCustomerFirstname($quote->getBillingAddress()->getFirstname());
                $quote->setCustomerLastname($quote->getBillingAddress()->getLastname());
                if ($quote->getBillingAddress()->getMiddlename() === null) {
                    $quote->setCustomerMiddlename($quote->getBillingAddress()->getMiddlename());
                }
            }
            $quote->setCustomerIsGuest(true);
            $groupId = $quote->getCustomer()->getGroupId() ?: GroupInterface::NOT_LOGGED_IN_ID;
            $quote->setCustomerGroupId($groupId);
        }

        $remoteAddress = $this->remoteAddress->getRemoteAddress();
        if ($remoteAddress !== false) {
            $quote->setRemoteIp($remoteAddress);
            $quote->setXForwardedFor(
                $this->request->getServer('HTTP_X_FORWARDED_FOR')
            );
        }

        $this->_eventManager->dispatch('checkout_submit_before', ['quote' => $quote]);

//        $order = $this->submit($quote);
        // Place Order
        $order = $this->placeOrderHelper->submit($quote);

        if (null == $order) {
            throw new LocalizedException(
                __('A server error stopped your order from being placed. Please try to place your order again.')
            );
        }

        $this->checkoutSession->setLastQuoteId($quote->getId());
        $this->checkoutSession->setLastSuccessQuoteId($quote->getId());
        $this->checkoutSession->setLastOrderId($order->getId());
        $this->checkoutSession->setLastRealOrderId($order->getIncrementId());
        $this->checkoutSession->setLastOrderStatus($order->getStatus());

        $this->_eventManager->dispatch('checkout_submit_all_after', ['order' => $order, 'quote' => $quote]);
        return $order->getId();
    }

    /**
     * Submit quote
     *
     * @param Quote $quote
     * @param array $orderData
     * @return \Magento\Framework\Model\AbstractExtensibleModel|\Magento\Sales\Api\Data\OrderInterface|object|null
     * @throws \Exception
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function submit(QuoteEntity $quote, $orderData = [])
    {
        if (!$quote->getAllVisibleItems()) {
            $quote->setIsActive(false);
            return null;
        }
        return $this->submitQuote($quote, $orderData);
    }

    protected function submitQuote(QuoteEntity $quote, $orderData = [])
    {
        $order = $this->orderInterfaceFactory->create();
//        $this->submitQuoteValidator->validateQuote($quote);
        if (!$quote->getCustomerIsGuest()) {
            if ($quote->getCustomerId()) {
                $this->_prepareCustomerQuote($quote);
                $this->customerManagement->validateAddresses($quote);
            }
            $this->customerManagement->populateCustomerInfo($quote);
        }
        $addresses = [];
        $quote->reserveOrderId();
        if ($quote->isVirtual()) {
            $this->dataObjectHelper->mergeDataObjects(
                \Magento\Sales\Api\Data\OrderInterface::class,
                $order,
                $this->quoteAddressToOrder->convert($quote->getBillingAddress(), $orderData)
            );
        } else {
            $this->dataObjectHelper->mergeDataObjects(
                \Magento\Sales\Api\Data\OrderInterface::class,
                $order,
                $this->quoteAddressToOrder->convert($quote->getShippingAddress(), $orderData)
            );
            $shippingAddress = $this->quoteAddressToOrderAddress->convert(
                $quote->getShippingAddress(),
                [
                    'address_type' => 'shipping',
                    'email' => $quote->getCustomerEmail()
                ]
            );
            $shippingAddress->setData('quote_address_id', $quote->getShippingAddress()->getId());
            $addresses[] = $shippingAddress;
            $order->setShippingAddress($shippingAddress);
            $order->setShippingMethod($quote->getShippingAddress()->getShippingMethod());
        }
        $billingAddress = $this->quoteAddressToOrderAddress->convert(
            $quote->getBillingAddress(),
            [
                'address_type' => 'billing',
                'email' => $quote->getCustomerEmail()
            ]
        );
        $billingAddress->setData('quote_address_id', $quote->getBillingAddress()->getId());
        $addresses[] = $billingAddress;
        $order->setBillingAddress($billingAddress);
        $order->setAddresses($addresses);
        $order->setPayment($this->quotePaymentToOrderPayment->convert($quote->getPayment()));
        $order->setItems($this->resolveItems($quote));
        if ($quote->getCustomer()) {
            $order->setCustomerId($quote->getCustomer()->getId());
        }
        $order->setQuoteId($quote->getId());
        $order->setCustomerEmail($quote->getCustomerEmail());
        $order->setCustomerFirstname($quote->getCustomerFirstname());
        $order->setCustomerMiddlename($quote->getCustomerMiddlename());
        $order->setCustomerLastname($quote->getCustomerLastname());
//        $this->submitQuoteValidator->validateOrder($order);

        $this->_eventManager->dispatch(
            'sales_model_service_quote_submit_before',
            [
                'order' => $order,
                'quote' => $quote
            ]
        );
        try {
            $order = $this->orderManagement->place($order);
//            $quote->setIsActive(true);
//            $this->_eventManager->dispatch(
//                'sales_model_service_quote_submit_success',
//                [
//                    'order' => $order,
//                    'quote' => $quote
//                ]
//            );
            $this->quoteRepository->save($quote);
        } catch (\Exception $e) {
            $this->rollbackAddresses($quote, $order, $e);
            throw $e;
        }
        return $order;
    }

    /**
     * Prepare address for customer quote.
     *
     * @param Quote $quote
     * @return void
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     */
    protected function _prepareCustomerQuote($quote)
    {
        /** @var Quote $quote */
        $billing = $quote->getBillingAddress();
        $shipping = $quote->isVirtual() ? null : $quote->getShippingAddress();

        $customer = $this->customerRepository->getById($quote->getCustomerId());
        $hasDefaultBilling = (bool)$customer->getDefaultBilling();
        $hasDefaultShipping = (bool)$customer->getDefaultShipping();

        if ($shipping && !$shipping->getSameAsBilling()
            && (!$shipping->getCustomerId() || $shipping->getSaveInAddressBook())
        ) {
            if ($shipping->getQuoteId()) {
                $shippingAddress = $shipping->exportCustomerAddress();
            } else {
                $defaultShipping = $this->customerRepository->getById($customer->getId())->getDefaultShipping();
                if ($defaultShipping) {
                    try {
                        $shippingAddress = $this->addressRepository->getById($defaultShipping);
                        // phpcs:ignore Magento2.CodeAnalysis.EmptyBlock
                    } catch (LocalizedException $e) {
                        // no address
                    }
                }
            }
            if (isset($shippingAddress)) {
                if (!$hasDefaultShipping) {
                    //Make provided address as default shipping address
                    $shippingAddress->setIsDefaultShipping(true);
                    $hasDefaultShipping = true;
                    if (!$hasDefaultBilling && !$billing->getSaveInAddressBook()) {
                        $shippingAddress->setIsDefaultBilling(true);
                        $hasDefaultBilling = true;
                    }
                }
                //save here new customer address
                $shippingAddress->setCustomerId($quote->getCustomerId());
                $this->addressRepository->save($shippingAddress);
                $quote->addCustomerAddress($shippingAddress);
                $shipping->setCustomerAddressData($shippingAddress);
                $this->addressesToSync[] = $shippingAddress->getId();
                $shipping->setCustomerAddressId($shippingAddress->getId());
            }
        }

        if (!$billing->getCustomerId() || $billing->getSaveInAddressBook()) {
            if ($billing->getQuoteId()) {
                $billingAddress = $billing->exportCustomerAddress();
            } else {
                $defaultBilling = $this->customerRepository->getById($customer->getId())->getDefaultBilling();
                if ($defaultBilling) {
                    try {
                        $billingAddress = $this->addressRepository->getById($defaultBilling);
                        // phpcs:ignore Magento2.CodeAnalysis.EmptyBlock
                    } catch (LocalizedException $e) {
                        // no address
                    }
                }
            }
            if (isset($billingAddress)) {
                if (!$hasDefaultBilling) {
                    //Make provided address as default shipping address
                    if (!$hasDefaultShipping) {
                        //Make provided address as default shipping address
                        $billingAddress->setIsDefaultShipping(true);
                    }
                    $billingAddress->setIsDefaultBilling(true);
                }
                $billingAddress->setCustomerId($quote->getCustomerId());
                $this->addressRepository->save($billingAddress);
                $quote->addCustomerAddress($billingAddress);
                $billing->setCustomerAddressData($billingAddress);
                $this->addressesToSync[] = $billingAddress->getId();
                $billing->setCustomerAddressId($billingAddress->getId());
            }
        }
        if ($shipping && !$shipping->getCustomerId() && !$hasDefaultBilling) {
            $shipping->setIsDefaultBilling(true);
        }
    }

    /**
     * Convert quote items to order items for quote
     *
     * @param Quote $quote
     * @return array
     */
    protected function resolveItems(QuoteEntity $quote)
    {
        $orderItems = [];
        foreach ($quote->getAllItems() as $quoteItem) {
            $itemId = $quoteItem->getId();

            if (!empty($orderItems[$itemId])) {
                continue;
            }

            $parentItemId = $quoteItem->getParentItemId();
            /** @var \Magento\Quote\Model\ResourceModel\Quote\Item $parentItem */
            if ($parentItemId && !isset($orderItems[$parentItemId])) {
                $orderItems[$parentItemId] = $this->quoteItemToOrderItem->convert(
                    $quoteItem->getParentItem(),
                    ['parent_item' => null]
                );
            }
            $parentItem = isset($orderItems[$parentItemId]) ? $orderItems[$parentItemId] : null;
            $orderItems[$itemId] = $this->quoteItemToOrderItem->convert($quoteItem, ['parent_item' => $parentItem]);
        }
        return array_values($orderItems);
    }

    /**
     * Remove related to order and quote addresses and submit exception to further processing.
     *
     * @param Quote $quote
     * @param \Magento\Sales\Api\Data\OrderInterface $order
     * @param \Exception $e
     * @throws \Exception
     */
    private function rollbackAddresses(
        QuoteEntity $quote,
        \Magento\Sales\Api\Data\OrderInterface $order,
        \Exception $e
    ): void {
        try {
            if (!empty($this->addressesToSync)) {
                foreach ($this->addressesToSync as $addressId) {
                    $this->addressRepository->deleteById($addressId);
                }
            }
            $this->_eventManager->dispatch(
                'sales_model_service_quote_submit_failure',
                [
                    'order' => $order,
                    'quote' => $quote,
                    'exception' => $e,
                ]
            );
        } catch (\Exception $consecutiveException) {
            $message = sprintf(
                "An exception occurred on 'sales_model_service_quote_submit_failure' event: %s",
                $consecutiveException->getMessage()
            );
            // phpcs:ignore Magento2.Exceptions.DirectThrow
            throw new \Exception($message, 0, $e);
        }
    }

}
