<?php

namespace UseePay\Payments\Controller\Payment;

use Magento\Framework\App\Action\Context;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Framework\App\RequestInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Email\Sender\OrderSender;
use Magento\Sales\Model\OrderFactory;
use UseePay\Payments\service\InvoiceService;
use UseePay\Payments\Controller\AbsUseePayAction;
use UseePay\Payments\Model\Adminhtml\Source\Config;
use UseePay\Payments\Helper\SignatureHelper;
use UseePay\Payments\Helper\OrderStatusHelper;
use UseePay\Payments\service\LockService;

class AsyncResult extends AbsUseePayAction implements CsrfAwareActionInterface
{
    /**
     * @var Config
     */
    private $config;
    /**
     * @var OrderFactory
     */
    private $orderFactory;
    /**
     * @var OrderSender
     */
    private $orderSender;

    const TRANSACTION_TYPE_ORDER = 'order';

    /**
     * @var \Magento\Quote\Api\CartRepositoryInterface
     */
    protected $quoteRepository;

    protected $lockService;
    /**
     * @var InvoiceService
     */
    private $invoiceService;


    public function __construct(Context $context, Config $config, OrderFactory $orderFactory,
                                OrderSender $orderSender, \Magento\Quote\Api\CartRepositoryInterface $quoteRepository = null,
                                LockService $lockService,
                                InvoiceService $invoiceService)
    {
        parent::__construct($context);
        $this->config       = $config;
        $this->orderFactory = $orderFactory;
        $this->orderSender  = $orderSender;
        $this->quoteRepository = $quoteRepository ?: ObjectManager::getInstance()
            ->get(\Magento\Quote\Api\CartRepositoryInterface::class);
        $this->lockService    = $lockService;
        $this->invoiceService = $invoiceService;
    }

    public function execute()
    {
        // 记录日志
        $result = $_POST;
        $this->saveLog('response-AsyncResult:');
        $this->saveLog(var_export($result, true));
        $this->saveLog('------------------------------');

        $order = $this->orderFactory->create()->loadByIncrementId($result['transactionId']);
        if (SignatureHelper::verifyMd5Signature($result, $this->config->getMd5SecretKey())) {
            $status_history = $order->getState();
            if (OrderStatusHelper::isSucceed($result)) {
                $this->paySuccess($order, $result, $status_history);
            } elseif (OrderStatusHelper::isFailed($result)) {
                $this->payFailed($status_history, $order, $result);
            } else {
                $order->addStatusToHistory(Order::STATE_PAYMENT_REVIEW, __('payment is processing, Result Message is: ' . $result["errorMsg"]));
            }
        } else {
            $this->saveLog('Invalid signature!');
        }
        echo 'OK';
    }

    public function createOrderInvoice($order, $result)
    {
        if ($order->canInvoice()) {
            $invoice = $order->prepareInvoice();
            if ($invoice->getTotalQty()) {
                $invoice->register();
                $invoice->setState(\Magento\Sales\Model\Order\Invoice::STATE_PAID);
                $invoice->save();
                $order->addRelatedObject($invoice);
            }

            $transactionSave = $this->_objectManager->create('Magento\Framework\DB\Transaction')
                ->addObject($invoice)
                ->addObject($invoice->getOrder());
            $transactionSave->save();

            $order->addStatusHistoryComment(__('Created invoice #%1,reference :'.$result['reference'], $invoice->getId()))->setIsCustomerNotified(true)->save();

            $this->createTransaction($order, $result);
        }
    }

    public function createTransaction($order, $result)
    {
        $payment = $this->_objectManager->create('Magento\Sales\Model\Order\Payment');
        $payment->setTransactionId($result['reference']);
        $payment->setOrder($order);
        $payment->setIsTransactionClosed(1);
        $transaction = $payment->addTransaction(self::TRANSACTION_TYPE_ORDER);
        $transaction->beforeSave();
        $transaction->save();
    }

    public function createCsrfValidationException(RequestInterface $request): ?InvalidRequestException
    {
        return null;
    }

    public function validateForCsrf(RequestInterface $request): ?bool
    {
        return true;
    }

    /**
     * @param $order
     * @param array $result
     * @param $status_history
     * @return void
     */
    public function paySuccess($order, array $result, $status_history): void
    {
        $quote = $this->quoteRepository->get($order->getQuoteId())->setIsActive(false);
        $this->quoteRepository->save($quote);
        if (OrderStatusHelper::canChangeStatus($status_history)) {
            $lock = $this->lockService->lock($result['transactionId']);
            try {
                if ($lock) {
                    $order = $this->orderFactory->create()->loadByIncrementId($result['transactionId']);
                    $status_history = $order->getState();
                    if (OrderStatusHelper::canChangeStatus($status_history)) {
                        $order->getPayment()
                            ->setPreparedMessage('')
                            ->setTransactionId($result['reference'])
                            ->setIsTransactionClosed(0)
                            ->update(false);
                        $order->setState(Order::STATE_PROCESSING);
                        $order->setStatus(Order::STATE_PROCESSING);
                        $order->addStatusToHistory(Order::STATE_PROCESSING, __('payment is successful.'));
                        $order->setTotalPaid($order->getGrandTotal());
                        $order->setBaseTotalPaid($order->getGrandTotal());
                        $order->save();
                        $this->invoiceService->createOrderInvoice($order, $result);
                        $this->orderSender->send($order);
                    }
                }
            } finally {
                if ($lock) {
                    $this->lockService->unlock($result['transactionId']);
                }
            }
        }
    }

    /**
     * @param $status_history
     * @param $order
     * @param $errorMsg
     * @return void
     */
    public function payFailed($status_history, $order, $result): void
    {
        $errorMsg  = $result['errorMsg'];
        if (OrderStatusHelper::canChangeFailStatus($status_history)) {
            $lock = $this->lockService->lock($result['transactionId']);
            try {
                if ($lock) {
                    $order->setTransactionId($result['reference']);
                    $order->setState(Order::STATE_CANCELED);
                    $order->setStatus(Order::STATE_CANCELED);
                    $order->addStatusToHistory(Order::STATE_CANCELED, __('sorry, payment is failed! Result Message is: ' . $errorMsg));
                    $order->save();
                }
            } finally {
                if ($lock) {
                    $this->lockService->unlock($result['transactionId']);
                }
            }
        }
    }
}
